<?php

namespace CleantalkSP\SpbctWP\VulnerabilityAlarm;

use CleantalkSP\SpbctWP\VulnerabilityAlarm\Dto\ItemReport;
use CleantalkSP\SpbctWP\VulnerabilityAlarm\Dto\PluginReport;

class VulnerabilityAlarmView
{
    const ASSETS_URL = SPBC_PATH . '/lib/CleantalkSP/SpbctWP/VulnerabilityAlarm/Assets';
    const RESEARCH_SITE_URL = 'https://research.cleantalk.org';

    /**
     * @param string $plugin_file
     * @param PluginReport $plugin_report
     *
     * @return string
     */
    public static function showPluginAlarm($plugin_file, $plugin_report)
    {
        $active = is_plugin_active($plugin_file) ? 'active' : 'inactive';
        $slug = $plugin_report->slug;
        $id = $plugin_report->id;
        $alarm_text = esc_html__('This version contains a known vulnerability', 'security-malware-firewall');
        $alarm_cve_url = $plugin_report->CVE;
        $delete_text = esc_html__('It is strongly recommended to delete the plugin.', 'security-malware-firewall');
        $update_text = esc_html__('Updating the plugin to a version higher than ', 'security-malware-firewall');

        $vulnerability_text = sprintf(
            $alarm_text . ' ' . $alarm_cve_url . '. %s',
            ! empty($plugin_report->rs_app_version_max)
                ? $update_text . $plugin_report->rs_app_version_max . ' is strongly recommended.'
                : $delete_text
        );

        $layout_file = __DIR__ . '/View/PluginListAlarmLayout.html';

        if ( file_exists($layout_file) ) {
            $replaces = [
                '{{SLUG}}' => esc_attr($slug),
                '{{ACTIVE}}' => esc_attr($active),
                '{{PLUGIN_FILE}}' => esc_attr($plugin_file),
                '{{VULNERABILITY_TEXT}}' => wp_kses($vulnerability_text, 'post'),
                '{{MORE_DETAILS_LINK}}' => static::getMoreDetailsLink($slug, $id)
            ];
            $layout = file_get_contents($layout_file);

            if ($layout) {
                foreach ($replaces as $place_holder => $replace) {
                    $layout = str_replace($place_holder, $replace, $layout);
                }
                return $layout;
            }
        }

        return '<tr><td colspan="4">' . $vulnerability_text . '</td></tr>';
    }

    /**
     * Displays an alarm for a theme identified as vulnerable.
     *
     * This method generates an alarm message for a theme that has been identified as having a known vulnerability.
     * It constructs a message detailing the vulnerability and, depending on the severity, recommends either updating
     * the theme to a specific version or deleting it entirely. The message is then inserted into a predefined HTML
     * layout for themes, if available, or returned as a simple paragraph element.
     *
     * @param object $theme_report An object containing the theme's details, including its slug, ID, CVE, and the maximum
     *                             safe version of the theme, if applicable.
     * @return string The generated alarm message wrapped in either the predefined HTML layout or a paragraph element.
     */
    public static function showThemeAlarm($theme_report)
    {
        $slug = $theme_report->slug;
        $id = $theme_report->id;
        $alarm_text = esc_html__('This theme version contains a known vulnerability', 'security-malware-firewall');
        $alarm_cve_url = $theme_report->CVE;
        $delete_text = esc_html__('It is strongly recommended to delete the theme.', 'security-malware-firewall');
        $update_text = esc_html__('Updating the theme to a version higher than ', 'security-malware-firewall');

        $vulnerability_text = sprintf(
            $alarm_text . ' ' . $alarm_cve_url . '. %s',
            ! empty($theme_report->rs_app_version_max)
                ? $update_text . $theme_report->rs_app_version_max . ' is strongly recommended.'
                : $delete_text
        );

        $layout_file = __DIR__ . '/View/ThemesListAlarmLayout.html';

        if ( file_exists($layout_file) && is_readable($layout_file)) {
            $replaces = [
                '{{SLUG}}' => esc_attr($slug),
                '{{VULNERABILITY_TEXT}}' => wp_kses($vulnerability_text, 'post'),
                '{{MORE_DETAILS_LINK}}' => static::getMoreDetailsLink($slug, $id)
            ];
            $layout = @file_get_contents($layout_file);

            if ($layout) {
                foreach ($replaces as $place_holder => $replace) {
                    $layout = str_replace($place_holder, $replace, $layout);
                }
                return $layout;
            }
        }

        return '<p>' . $vulnerability_text . '</p>';
    }

    public static function showSafeBadgeCMWS($module_type, $plugin_slug, $plugin_id)
    {
        $badge_text = esc_html__('Plugin is safe', 'security-malware-firewall');
        $module_type_translated = esc_html__('plugin', 'security-malware-firewall');

        if ($module_type === 'theme') {
            $badge_text = esc_html__('Theme is safe', 'security-malware-firewall');
            $module_type_translated = esc_html__('theme', 'security-malware-firewall');
        }

        $layout_file = __DIR__ . '/View/PluginIsSafeBadge.html';

        if ( file_exists($layout_file) ) {
            $replaces = [
                '{{ASSETS_URL}}' => self::ASSETS_URL,
                '{{DESCRIPTION}}' => static::getSafeReportChunkCMWS($plugin_slug, $plugin_id, $module_type_translated),
                '{{MORE_DETAILS_LINK}}' => static::getVASupportChunk(),
            ];
            $layout = file_get_contents($layout_file);

            foreach ($replaces as $place_holder => $replace) {
                $layout = str_replace($place_holder, $replace, $layout);
            }

            return $layout;
        }

        return $badge_text;
    }

    public static function showSafeBadgePSC($module_type, $plugin_slug, $plugin_id, $psc_cert_name)
    {

        $badge_text = esc_html__('Plugin is PSC certified', 'security-malware-firewall');
        $module_type_translated = esc_html__('plugin', 'security-malware-firewall');

        if ($module_type === 'theme') {
            $badge_text = esc_html__('Theme is PSC certified', 'security-malware-firewall');
            $module_type_translated = esc_html__('theme', 'security-malware-firewall');
        }

        $layout_file = __DIR__ . '/View/PluginIsSafePSCBadge.html';

        if ( file_exists($layout_file) ) {
            $replaces = [
                '{{ASSETS_URL}}' => self::ASSETS_URL,
                '{{DESCRIPTION}}' => static::getSafeReportChunkPSC($plugin_slug, $plugin_id, $psc_cert_name, $module_type_translated),
                '{{MORE_DETAILS_LINK}}' => static::getVASupportChunk(),
            ];
            $layout = file_get_contents($layout_file);

            foreach ($replaces as $place_holder => $replace) {
                $layout = str_replace($place_holder, $replace, $layout);
            }

            return $layout;
        }

        return $badge_text;
    }

    /**
     * @param string $module_slug
     * @param string $module_id
     * @param bool $line_break Should the lines be broken
     *
     * @return string
     */
    private static function getMoreDetailsLink($module_slug = '', $module_id = '', $line_break = false)
    {
        $line_break_layout = $line_break ? '<br>' : ' ';
        return self::getFullReportChunk($module_slug, $module_id)
            . $line_break_layout
            . self::getVASupportChunk();
    }

    /**
     * Returns support link and description.
     * @return string
     */
    private static function getVASupportChunk()
    {
        return sprintf(
            esc_html__('Have questions? Ask us %shere%s.', 'security-malware-firewall'),
            '<a href="https://wordpress.org/support/plugin/security-malware-firewall/" target="_blank">',
            '</a>'
        );
    }

    /**
     * @param $module_slug
     * @param $module_id
     * @return string
     */
    private static function getFullReportChunk($module_slug, $module_id)
    {
        $details_link = static::RESEARCH_SITE_URL;
        if ($module_slug && $module_id) {
            $details_link = esc_attr($details_link . '/reports/app/' . $module_slug . '#' . $module_id);
        }
        $report_text = esc_html__('The full report is %shere%s.', 'security-malware-firewall');
        return sprintf(
            $report_text,
            '<a href="' . $details_link . '" target="_blank">',
            '</a>'
        );
    }

    /**
     * @param $module_slug
     * @param $module_id
     * @param $psc_cert_name
     * @param $module_type_translated
     * @return string
     */
    private static function getSafeReportChunkPSC($module_slug, $module_id, $psc_cert_name, $module_type_translated)
    {
        $details_link = static::RESEARCH_SITE_URL;
        if ($module_slug && $module_id) {
            $details_link = esc_attr($details_link . '/reports/app/' . $module_slug . '#' . $module_id);
        }
        $report_text = esc_html__('This version of the %s is security safe and certified as %s%s%s', 'security-malware-firewall');
        return sprintf(
            $report_text,
            $module_type_translated,
            '<a href="' . $details_link . '" target="_blank">',
            $psc_cert_name,
            '</a>'
        );
    }

    /**
     * @param $module_slug
     * @param $module_id
     * @param $module_type_translated
     * @return string
     */
    private static function getSafeReportChunkCMWS($module_slug, $module_id, $module_type_translated)
    {
        $details_link = static::RESEARCH_SITE_URL;
        if ($module_slug && $module_id) {
            $details_link = esc_attr($details_link . '/reports/app/' . $module_slug . '#' . $module_id);
        }
        $report_text = esc_html__('This version of the %s is security safe and does not have known vulnerabilities.%sCloud Malware Scanner by CleanTalk, %sfull report%s', 'security-malware-firewall');
        return sprintf(
            $report_text,
            $module_type_translated,
            '<br>',
            '<a href="' . $details_link . '" target="_blank">',
            '</a>'
        );
    }

    /**
     * Render tab for spbct settings page
     */
    public static function renderSettingsTab()
    {
        $layout_file = __DIR__ . '/View/VulnerabilityAlarmTab.html';
        $layout_item_file = __DIR__ . '/View/VulnerabilityAlarmTabItem.html';

        if ( file_exists($layout_file) && file_exists($layout_item_file) ) {
            $vulnerable_plugins = static::getVulnerablePluginsStatic();
            $vulnerable_themes = static::getVulnerableThemesStatic();

            $known_vulnerabilities_title = __('Known vulnerabilities', 'security-malware-firewall');
            $replaces = [
                '{{VULNERABILITIES_TITLE}}' => $known_vulnerabilities_title,
                '{{VULNERABILITIES_SUBTITLE}}' =>
                    count($vulnerable_plugins) > 0 || count($vulnerable_themes) > 0
                        ? __('Vulnerabilities contains in: ', 'security-malware-firewall')
                        : __('No vulnerabilities found', 'security-malware-firewall'),
            ];

            $layout = file_get_contents($layout_file);
            $item_layout = file_get_contents($layout_item_file);

            foreach ($replaces as $place_holder => $replace) {
                $layout = str_replace($place_holder, $replace, $layout);
            }

            $vulnerable_items_layout = '';

            if ( count($vulnerable_plugins) > 0 ) {
                foreach ($vulnerable_plugins as $plugin_report ) {
                    $delete_text = esc_html__('It is strongly recommended to delete the plugin.', 'security-malware-firewall');
                    $update_text = esc_html__('Updating the plugin to a version higher than ', 'security-malware-firewall');
                    $cve_url = esc_attr('https://research.cleantalk.org/reports/app/' . $plugin_report->slug . '#' . $plugin_report->id);
                    $cve_id = !empty($plugin_report->CVE) ? $plugin_report->CVE : 'CVE-' . $plugin_report->id;
                    $cve_url_link = sprintf('<a href="%s" target="_blank">%s</a>', $cve_url, $cve_id);
                    $vulnerability_text = sprintf(
                        __('contains', 'security-malware-firewall') . ' ' . $cve_url_link . '. %s',
                        ! empty($plugin_report->rs_app_version_max)
                            ? $update_text . $plugin_report->rs_app_version_max . esc_html__(' is strongly recommended.', 'security-malware-firewall')
                            : $delete_text
                    );
                    $vulnerability_versions_template = '(%s -> %s)';
                    $version_min = !empty($plugin_report->rs_app_version_min) ? $plugin_report->rs_app_version_min : '0.0';
                    $version_max = !empty($plugin_report->rs_app_version_max) ? $plugin_report->rs_app_version_max : 'unknown';
                    $vulnerability_versions = sprintf($vulnerability_versions_template, $version_min, $version_max);
                    $clickable_plugin_name = '<a href="plugins.php#%s" target="_blank">%s</a>';
                    $clickable_plugin_name = sprintf(
                        $clickable_plugin_name,
                        $plugin_report->slug . '-spbc-anchor',
                        esc_html($plugin_report->app_name ?: $plugin_report->slug)
                    );
                    $plugin_item_replaces = [
                        '{{VULNERABLE_NAME}}' => $clickable_plugin_name,
                        '{{VULNERABLE_VERSIONS}}' => esc_html__(' of version ', 'security-malware-firewall') .  $vulnerability_versions,
                        '{{VULNERABLE_ALERT}}' => $vulnerability_text,
                    ];
                    $vulnerable_items_layout .= str_replace(array_keys($plugin_item_replaces), array_values($plugin_item_replaces), $item_layout);
                }
            }

            if ( count($vulnerable_themes) > 0 ) {
                foreach ( $vulnerable_themes as $theme_report ) {
                    $delete_text = esc_html__('It is strongly recommended to delete the theme.', 'security-malware-firewall');
                    $update_text = esc_html__('Updating the theme to a version higher than ', 'security-malware-firewall');
                    $cve_url = esc_attr('https://research.cleantalk.org/reports/app/' . $theme_report->slug . '#' . $theme_report->id);
                    $cve_id = !empty($theme_report->CVE) ? $theme_report->CVE : 'CVE-' . $theme_report->id;
                    $cve_url_link = sprintf('<a href="%s" target="_blank">%s</a>', $cve_url, $cve_id);
                    $vulnerability_text = sprintf(
                        __('contains', 'security-malware-firewall') . ' ' . $cve_url_link . '. %s',
                        ! empty($theme_report->rs_app_version_max)
                            ? $update_text . $theme_report->rs_app_version_max . esc_html__(' is strongly recommended.', 'security-malware-firewall')
                            : $delete_text
                    );
                    $vulnerability_versions_template = '(%s -> %s)';
                    $version_min = !empty($plugin_report->rs_app_version_min) ? $theme_report->rs_app_version_min : '0.0';
                    $version_max = !empty($plugin_report->rs_app_version_max) ? $theme_report->rs_app_version_max : 'unknown';
                    $vulnerability_versions = sprintf($vulnerability_versions_template, $version_min, $version_max);
                    $clickable_theme_name = '<a href="themes.php" target="_blank">%s</a>';
                    $clickable_theme_name = sprintf(
                        $clickable_theme_name,
                        esc_html($theme_report->app_name ?: $theme_report->slug)
                    );
                    $theme_item_replaces = [
                        '{{VULNERABLE_NAME}}' => $clickable_theme_name,
                        '{{VULNERABLE_VERSIONS}}' => esc_html__(' of version ', 'security-malware-firewall') . $vulnerability_versions,
                        '{{VULNERABLE_ALERT}}' => $vulnerability_text,
                    ];
                    $vulnerable_items_layout .= str_replace(array_keys($theme_item_replaces), array_values($theme_item_replaces), $item_layout);
                }
            }

            $layout = str_replace('{{VULNERABLE_ITEMS}}', $vulnerable_items_layout, $layout);

            // PSC
            $psc_title = __('PSC certified plugins/themes', 'security-malware-firewall');
            $psc_modules = static::getPscModules();
            $psc_items_layout = '';
            $psc_replaces = [
                '{{PSC_TITLE}}' => __('PSC certified plugins/themes', 'security-malware-firewall'),
                '{{PSC_SUBTITLE}}' =>
                    count($psc_modules) > 0
                        ? __('The following plugins/themes are PSC certified:', 'security-malware-firewall')
                        : __('No PSC certified plugins/themes found', 'security-malware-firewall'),
            ];
            foreach ($psc_replaces as $place_holder => $replace) {
                $layout = str_replace($place_holder, $replace, $layout);
            }
            if ( count($psc_modules) > 0 ) {
                foreach ($psc_modules as $module ) {
                    $psc_url = esc_attr('https://research.cleantalk.org/reports/app/' . $module['slug']);
                    $psc_url_link = sprintf('<a href="%s" target="_blank">%s</a>', $psc_url, $module['psc']);
                    $psc_item_replaces = [
                        '{{VULNERABLE_NAME}}' => $module['slug'],
                        '{{VULNERABLE_VERSIONS}}' => ' of version ' . $module['version'],
                        '{{VULNERABLE_ALERT}}' => ' has ' . $psc_url_link,
                    ];
                    $psc_items_layout .= str_replace(array_keys($psc_item_replaces), array_values($psc_item_replaces), $item_layout);
                }
            }
            $layout = str_replace('{{PSC_ITEMS}}', $psc_items_layout, $layout);

            //Legend
            $legend_item_template = '%s - %s';
            $full_db_href = '<a href="https://research.cleantalk.org/reports/" target="_blank">https://research.cleantalk.org/reports/</a>';
            $legend_items_replace = [
                '{{DESCRIPTION_FEATURE}}' =>
                    static::getServiceFeatureDescription()
                ,
                '{{DESCRIPTION_LEGEND}}' => __('Categories description:', 'security-malware-firewall')

                ,
                '{{LEGEND_KNOWN_V}}' => sprintf(
                    $legend_item_template,
                    $known_vulnerabilities_title,
                    __('plugins ant themes on your website with known and published vulnerabilities.', 'security-malware-firewall')
                ),
                '{{LEGEND_PSC}}' => sprintf(
                    $legend_item_template,
                    $psc_title,
                    __('tested by Cloud Malware scanner or CleanTalk Research team and do not have known security issues or vulnerabilities. These plugins and themes are safe to use by now.', 'security-malware-firewall')
                ),
                '{{LEGEND_DB_LINK}}' => sprintf(
                    '%s %s %s',
                    __('Full database', 'security-malware-firewall'),
                    __('of vulnerabilities and safe plugins/themes is here ', 'security-malware-firewall'),
                    $full_db_href
                ),
            ];

            $layout = str_replace(array_keys($legend_items_replace), array_values($legend_items_replace), $layout);

            //last update
            $last_update_replacements = static::getLastUpdateResultReplacements();
            $layout = str_replace(
                array(
                    '{{LAST_UPDATE_RESULT}}',
                    '{{CURRENT_STATE_CLASS}}',
                    '{{CURRENT_STATE_TEXT}}'
                ),
                array(
                    $last_update_replacements['last_update_result'],
                    $last_update_replacements['current_state_class'],
                    $last_update_replacements['current_state_text']
                ),
                $layout
            );

            //end
            return $layout;
        }

        return '';
    }

    /**
     * This method retrieves all vulnerable plugins in the application.
     *
     * It checks each plugin for any known vulnerabilities. If a vulnerability is found,
     * the plugin is considered vulnerable and its report is added to the list of vulnerable plugins.
     *
     * @return ItemReport[] An array containing the reports of all vulnerable plugins.
     */
    private static function getVulnerablePluginsStatic()
    {
        $plugins = get_plugins();
        $vulnerable_plugins = [];

        if ( count($plugins) > 0 ) {
            foreach ( $plugins as $plugin_data ) {
                $plugin_slug = isset($plugin_data['slug']) ? $plugin_data['slug'] : sanitize_title($plugin_data['Name']);
                $plugin_version = ! empty($plugin_data['Version']) ? $plugin_data['Version'] : '';
                $plugin_report = VulnerabilityAlarm::checkPluginVulnerabilityStatic($plugin_slug, $plugin_version);
                if ( $plugin_report ) {
                    $vulnerable_plugins[] = $plugin_report;
                }
            }
        }
        return $vulnerable_plugins;
    }

    /**
     * This method retrieves all vulnerable themes in the application.
     *
     * It checks each theme for any known vulnerabilities. If a vulnerability is found,
     * the theme is considered vulnerable and its report is added to the list of vulnerable themes.
     *
     * @return ItemReport[] An array containing the reports of all vulnerable themes.
     */
    private static function getVulnerableThemesStatic()
    {
        $themes = wp_get_themes();
        $vulnerable_themes = [];

        if ( count($themes) > 0 ) {
            foreach ( $themes as $theme_data ) {
                $theme_slug = isset($theme_data['slug']) ? $theme_data['slug'] : sanitize_title($theme_data['Name']);
                $theme_version = ! empty($theme_data['Version']) ? $theme_data['Version'] : '';
                $theme_report = VulnerabilityAlarm::checkSingleThemeVulnerabilityStatic($theme_slug, $theme_version);
                if ( $theme_report ) {
                    $vulnerable_themes[] = $theme_report;
                }
            }
        }
        return $vulnerable_themes;
    }


    /**
     * This method retrieves all outdated modules in the application.
     *
     * It checks both plugins and themes for any available updates. If an update is available,
     * it means the module is outdated and it's added to the list of outdated modules.
     *
     * @return array An array containing the names of all outdated modules (both plugins and themes).
     * @psalm-suppress UnusedMethod
     * todo This method is not used now, decision https://doboard.com/1/task/9092
     */
    private static function getOutdatedModules()
    {
        $all_plugins = get_plugins();
        $plugins_updates = get_plugin_updates();
        $all_themes = wp_get_themes();
        $themes_updates = get_theme_updates();

        $outdated_modules = [];

        // Check each plugin
        foreach ( $all_plugins as $plugin_path => $plugin ) {
            // If there is an update for the plugin
            if ( isset($plugins_updates[$plugin_path]) ) {
                $outdated_modules[] = $plugin['Name'];
            }
        }

        // Check each theme
        foreach ( $all_themes as $theme_path => $theme ) {
            // If there is an update for the theme
            if ( isset($themes_updates[$theme_path]) ) {
                $outdated_modules[] = $theme['Name'];
            }
        }

        return $outdated_modules;
    }

    /**
     * Retrieves a list of plugins certified by the Plugin Security Checker (PSC).
     *
     * This method first fetches all installed plugins using the `get_plugins()` function. It then filters these plugins
     * through the `getSafePlugins()` method to identify those that are considered safe. Among the safe plugins, it further
     * filters to identify those that have been specifically certified by the Plugin Security Checker (PSC), indicating
     * a higher level of security assurance. The PSC certification is determined by checking the `app_status` and `psc`
     * properties of each plugin's report data.
     *
     * @return array An array of plugins that are both considered safe and have PSC certification. Each element in the
     *               array is an associative array containing the plugin's ID, slug, and PSC certification status.
     */
    private static function getPscModules()
    {
        return VulnerabilityAlarm::getPSCSafePluginsOnly();
    }

    public static function getCountOfCurrnetlyVulnerableModules()
    {
        return count(static::getVulnerablePluginsStatic()) + count(static::getVulnerableThemesStatic());
    }

    /**
     * Get replacements for last update header.
     * @return array
     */
    public static function getLastUpdateResultReplacements()
    {
        global $spbc;

        $replacements = array();

        //collect total count
        $count_of_plugins = isset($spbc->scan_plugins_info['total_site_plugins_count'])
            ? $spbc->scan_plugins_info['total_site_plugins_count']
            : 0;
        $count_of_themes = isset($spbc->scan_themes_info['total_site_themes_count'])
            ? $spbc->scan_themes_info['total_site_themes_count']
            : 0;
        $total_found_modules = $count_of_plugins + $count_of_themes;

        //collect requested modules count
        $count_of_plugins_requested = isset($spbc->scan_plugins_info['total_site_plugins_count'])
            ? $spbc->scan_plugins_info['plugins_info_requested']
            : 0;
        $count_of_themes_requested = isset($spbc->scan_themes_info['total_site_themes_count'])
            ? $spbc->scan_themes_info['themes_info_requested']
            : 0;
        $total_modules_requested = $count_of_plugins_requested + $count_of_themes_requested;

        // collect vulnerable count
        $count_of_v_plugins = isset($spbc->scan_plugins_info['plugins_found_with_known_vulnerabilities'])
            ? $spbc->scan_plugins_info['plugins_found_with_known_vulnerabilities']
            : 0;
        $count_of_v_themes = isset($spbc->scan_themes_info['themes_found_with_known_vulnerabilities'])
            ? $spbc->scan_themes_info['themes_found_with_known_vulnerabilities']
            : 0;
        $total_found_modules_vulnerable = $count_of_v_plugins + $count_of_v_themes;

        //collect last update timestamp
        $date_ts = isset($spbc->data['spbc_security_check_vulnerabilities_last_call'])
            ? $spbc->data['spbc_security_check_vulnerabilities_last_call']
            : 0;

        //collect current count of modules vulnerabilities
        $current_vul_modules_count = static::getCountOfCurrnetlyVulnerableModules();

        //generate last update result
        if ($total_modules_requested && $date_ts) {
            $date = date('M d Y H:i:s', $date_ts + $spbc->data['site_utc_offset_in_seconds']);
            $replacements['last_update_result'] = sprintf(
                esc_html__('%s plugins and themes found, %s scanned, %s vulnerabilities found at %s on this site.', 'security-malware-firewall'),
                $total_found_modules,
                $total_modules_requested,
                $total_found_modules_vulnerable,
                $date
            );
        } else {
            $replacements['last_update_result'] = esc_html__('No last update data provided. The Security plugin will check plugins and themes soon.', 'security-malware-firewall');
        }

        //generate current state
        if ($current_vul_modules_count) {
            $replacements['current_state_text'] = sprintf(
                __('<b>%s vulnerabilities is still active!</b>', 'security-malware-firewall'),
                $current_vul_modules_count
            );
            $replacements['current_state_class'] = 'spbc-icon-lightbulb-red';
        } else {
            $replacements['current_state_text'] = sprintf(
                __('No active vulnerabilities found at the current state.', 'security-malware-firewall'),
                $current_vul_modules_count
            );
            $replacements['current_state_class'] = $total_modules_requested ? 'spbc-icon-lightbulb-green' : 'spbc-icon-lightbulb';
        }

        return $replacements;
    }

    public static function getServiceFeatureDescription()
    {
        $template = '
        <p>%s</p>
            <ul style="margin-left: 2%%; list-style: disc">
                <li>%s</li>
                <li>%s</li>
                <li>%s</li>
            </ul>
        <p>%s</p>
        ';
        $common_text = esc_html__('The Security plugin preforms a scheduled request to CleanTalk API to check every installed plugin or theme for known vulnerabilities. The report data are updated with the next triggers:', 'security-malware-firewall');
        $cron_module = esc_html__('within 100 seconds since any plugin or theme is activated', 'security-malware-firewall');
        $cron_self = esc_html__('within 30 seconds since the Security plugin is activated', 'security-malware-firewall');
        $cron_common = esc_html__('every 24 hours after last update', 'security-malware-firewall');
        $on_lists = esc_html__('Also the single check runs every time you observe plugins/themes install pages for every listed item.', 'security-malware-firewall');
        $template = sprintf(
            $template,
            $common_text,
            $cron_module,
            $cron_self,
            $cron_common,
            $on_lists
        );
        return $template;
    }
}
